#include <algorithm>
#include <cmath>
#include <iostream>
#include <queue>
#include <utility>
#include <vector>

using namespace std;

constexpr double CONNECTED = 25.4 * 2;
constexpr double EPS       = 1e-9;
constexpr double MAX_D = 160 / 2;

struct Model {
    double x, y, d;

    Model(double x, double y, double d) : x(x), y(y), d(d) {}

    bool operator<(const Model &other) const {
        if(x != other.x) return x < other.x;
        if(y != other.y) return y < other.y;
        return d < other.d;
    }

    friend ostream& operator<<(ostream& os, const Model& model) {
        return os << "Model(" << model.x << ", " << model.y << ", " << model.d << ")";
    }
};

int n;
vector<Model> m;

void rotateAll(double ax, double ay) {
    static constexpr double ANGLE = 1;
    for(auto& model : m) {
        double x = model.x - ax;
        double y = model.y - ay;
        model.x  = x * cos(ANGLE) - y * sin(ANGLE) + ax;
        model.y  = x * sin(ANGLE) + y * cos(ANGLE) + ay;
    }
}

bool areConnected(const Model& a, const Model& b) {
    double distance2 = (a.x - b.x) * (a.x - b.x) + (a.y - b.y) * (a.y - b.y);
    double connectedDistance = (a.d + b.d + CONNECTED);
    return distance2 <= (connectedDistance * connectedDistance + EPS);
}

bool bfs() {
    vector<int> visited(n, 0);
    queue<int> queue;

    visited[0] = 1;
    queue.push(0);

    int minConnected = (n >= 7) ? 2 : 1;

    while(!queue.empty()) {
        int current = queue.front();
        queue.pop();

        double minx = m[current].x - m[current].d - MAX_D - CONNECTED;
        double maxx = m[current].x + m[current].d + MAX_D + CONNECTED;

        auto lf = lower_bound(m.begin(), m.end(), Model(minx - EPS, 0, 0));
        auto rg = upper_bound(m.begin(), m.end(), Model(maxx + EPS, 0, 0));

        int connected = 0;
        while(lf != rg) {
            int index = lf - m.begin();
            if(current != index && areConnected(m[current], m[index])) {
                connected++;
                if (!visited[index]) {
                    visited[index] = 1;
                    queue.push(index);
                }
            }
            ++lf;
        }
        if(connected < minConnected) {
            return false;
        }
    }

    for(int v : visited) {
        if (v == 0) {
            return false;
        }
    }
    return true;
}

int main() {
    ios_base::sync_with_stdio(false);
    cin.tie(nullptr);

    cin >> n;
    m.reserve(n);

    double ax = 0, ay = 0;

    for(int i = 0; i < n; ++i) {
        double x, y, d;
        cin >> x >> y >> d;
        d /= 2;
        m.emplace_back(x, y, d);

        ax += x / n;
        ay += y / n;
    }

    if(n > 1000) {
        rotateAll(ax, ay);
    }

    sort(m.begin(), m.end());
    if (bfs()) {
        cout << "yes\n";
    } else {
        cout << "no\n";
    }

    return 0;
}
