#!/usr/bin/env python3

# generate branching (or linear) patterns with labels that follow different
# patterns

import argparse
import random
import sys

MAX_N = 1000000
MAX_LABEL = 1000000

def gen_random(nodes, parent):
    return (random.randint(0, MAX_LABEL), parent)

def gen_increasing(nodes, parent):
    return (nodes[parent][0] + 1, parent)

def gen_decreasing(nodes, parent):
    return (nodes[parent][0] - 1, parent)

def gen_alternating(nodes, parent):
    if parent == 0:
        return (nodes[parent][0] + 2, parent, False)
    if nodes[parent][2]:
        return (nodes[parent][0] + 3, parent, False)
    return (nodes[parent][0] - 1, parent, True)

def gen_increasing_decrease(nodes, parent):
    '''Count down from MAX_LABEL to MAX_LABEL - X, then again from MAX_LABEL
    to MAX_LABEL - X - 1, etc.'''
    if parent == 0:
        return (MAX_LABEL, parent, 10, 10)

    parent_label, parent_parent, left, prev_max = nodes[parent]

    if left == 0:
        return (MAX_LABEL, parent, prev_max + 1, prev_max + 1)
    return (parent_label - 1, parent, left - 1, prev_max)

def gen_increasing_increase(nodes, parent):
    '''Count up from MAX_LABEL - X to MAX_LABEL, 
    then again from MAX_LABEL - X - 1 to MAX_LABEL, etc.'''
    if parent == 0:
        return (MAX_LABEL - 10, parent, 10)

    parent_label, parent_parent, length = nodes[parent]

    if parent_label == MAX_LABEL:
        return (MAX_LABEL - length - 1, parent, length + 1)
    return (parent_label + 1, parent, length)

def gen_increasing_repeat(nodes, parent):
    '''Count up to some large value, repeatedly. Start at decreasing labels.'''

    RUNLENGTH = 50000

    if parent == 0:
        return (RUNLENGTH, parent, RUNLENGTH, RUNLENGTH)
    
    plabel, gparent, start, remaining = nodes[parent]

    if not remaining:
        return (start - 10, parent, start - 10, RUNLENGTH)

    return (plabel + 1, parent, start, remaining - 1)

def gen_decreasing_repeat(nodes, parent):
    '''Count down from some large value, repeatedly. Start at increasing
    labels.'''

    RUNLENGTH = 50000

    if parent == 0:
        return (RUNLENGTH, parent, RUNLENGTH, RUNLENGTH)
    
    plabel, gparent, start, remaining = nodes[parent]

    if not remaining:
        return (start + 10, parent, start + 10, RUNLENGTH)

    return (plabel - 1, parent, start, remaining - 1)

def main():
    generators = {
            'random': (gen_random, gen_random([], -1)[0]),
            'increasing': (gen_increasing, 0),
            'decreasing': (gen_decreasing, MAX_LABEL),
            'alternating': (gen_alternating, 0),
            'increasing_decrease': (gen_increasing_decrease, 0),
            'increasing_increase': (gen_increasing_increase, 0),
            'increasing_repeat': (gen_increasing_repeat, 0),
            'decreasing_repeat': (gen_decreasing_repeat, 0),
            }

    p = argparse.ArgumentParser()
    p.add_argument('-n', default=MAX_N, type=int, help='number of nodes')
    p.add_argument('-b', default=2, type=int, help='branching factor')
    p.add_argument('--pattern', choices=sorted(list(generators)), default='random')
    args = p.parse_args()

    def parent(i):
        return (i - 1) // args.b

    gen, initial_label = generators[args.pattern]

    # (label, parent, (possibly other information))
    nodes = [(initial_label, -1)]

    while len(nodes) < args.n:
        p = parent(len(nodes))
        nodes.append(gen(nodes, p))

    print(len(nodes))
    for n in nodes:
        print(n[0])

    for n in nodes[1:]:
        print(n[1] + 1)

if __name__ == '__main__':
    main()

